<?php

class ITSEC_Dashboard_Card_Malware_Scan extends ITSEC_Dashboard_Card {
	/**
	 * @inheritDoc
	 */
	public function get_slug() {
		return 'malware-scan';
	}

	/**
	 * @inheritDoc
	 */
	public function get_label() {
		return __( 'Malware Scans', 'it-l10n-ithemes-security-pro' );
	}

	/**
	 * @inheritDoc
	 */
	public function get_size() {
		$scan_only = ITSEC_Modules::get_setting( 'global', 'log_type' ) === 'file';

		return array(
			'minW'     => $scan_only ? 1 : 2,
			'minH'     => 2,
			'maxW'     => $scan_only ? 1 : 2,
			'maxH'     => $scan_only ? 2 : 3,
			'defaultW' => $scan_only ? 1 : 2,
			'defaultH' => 2,
		);
	}

	/**
	 * @inheritDoc
	 */
	public function query_for_data( array $query_args, array $settings ) {

		if ( isset( $query_args['period'] ) ) {
			$period = $query_args['period'];
		} else {
			$qa_schema = $this->get_query_args();
			$period    = $qa_schema['period']['default'];
		}

		if ( is_wp_error( $range = ITSEC_Dashboard_Util::_get_range( $period ) ) ) {
			return $range;
		}

		$entries = ITSEC_Log::get_entries( array(
			'__min_timestamp' => $range[0],
			'__max_timestamp' => $range[1],
			'module'          => array( 'malware', 'site-scanner' ),
			'type_not'        => array( 'process-start' ),
		), 100, 1, 'timestamp', 'DESC', array(
			'id',
			'code',
			'data',
			'module',
			'init_timestamp',
		) );

		if ( is_wp_error( $entries ) ) {
			return $entries;
		}

		$data = array();

		foreach ( $entries as $entry ) {
			$data[] = $this->format_log( $entry );
		}

		return $data;
	}

	/**
	 * @inheritDoc
	 */
	public function get_query_args() {
		$args = parent::get_query_args();

		$args['period'] = ITSEC_Dashboard_REST::get_period_arg();

		return $args;
	}

	/**
	 * @inheritdoc
	 */
	public function get_links() {
		return array(
			array(
				'rel'   => ITSEC_Dashboard_REST::LINK_REL . 'logs',
				'href'  => ITSEC_Core::get_logs_page_url( 'malware' ),
				'title' => __( 'View Logs', 'it-l10n-ithemes-security-pro' ),
				'media' => 'text/html',
				'cap'   => ITSEC_Core::get_required_cap(),
			),
			array(
				'rel'      => ITSEC_Dashboard_REST::LINK_REL . 'rpc',
				'title'    => __( 'Scan Now', 'it-l10n-ithemes-security-pro' ),
				'endpoint' => 'scan',
				'cap'      => ITSEC_Core::get_required_cap(),
				'callback' => array( $this, 'do_malware_scan' ),
			)
		);
	}

	/**
	 * Perform the malware scan.
	 *
	 * @return array|WP_Error
	 */
	public function do_malware_scan() {
		ITSEC_Modules::load_module_file( 'class-itsec-malware-scanner.php', 'malware' );
		$results = ITSEC_Malware_Scanner::scan();

		if ( is_wp_error( $results ) ) {
			return $results;
		}

		if ( is_array( $results ) ) {
			return array(
				'results' => $results,
			);
		}

		return new WP_Error( 'itsec-dashboard-card-malware-scan-unexpected-response', __( 'The scan request returned an unexpected response.', 'it-l10n-ithemes-security-pro' ) );
	}

	private function format_log( $entry ) {
		return array(
			'id'      => $entry['id'],
			'code'    => $entry['code'],
			'results' => $entry['data']['results'],
			'time'    => ITSEC_Lib::to_rest_date( $entry['init_timestamp'] ),
		);
	}
}
